use strict;
use Tree::Trie;

### Local vars ########################################################

my ($WordTrie, $WordHash, $CommandInfo);

$WordTrie = Etc('CommandWordTrie', Tree::Trie->new);
$WordHash = Etc('CommandWordHash', {}); 

my $cmd_allowed = sub {
  my ($con, $cmd) = @_;
  
  my $req = ($CommandInfo->get($cmd) or return 0)->{requires} or return 1;
  $con and $con->user or return 0;
  my $privs = $con->user->get('privileges') or return 0;
  return not grep !$privs->{$_}, @$req;
};
  

my $cmd_parse = sub {
  my ($line) = @_;

  $line =~ s#^\s*/?(@?[a-z]+|\S)(\s*)##i;
  return wantarray ? (lc $1, $line, $2) : lc $1;
};

my $cmd_split = sub {
  my ($line) = @_;

  my @segs = split /;|[\.!?]\s/, $line;
  my ($pcount, @cmds, $res) = (0);    
  foreach (@segs) {
    if ($pcount > 0) {
      $cmds[-1] .= ";$_";
    } else {
      push @cmds, $_;
    }
    $pcount += tr/(//;
    $pcount -= tr/)//;
  }

  return @cmds;
};

### Definitions #######################################################

Define '.' => {
  Verbs => MDefList->new(type => 'CODE'),
  Commands => $CommandInfo = MDefList->new(
    type => 'HASH',
    add_callback => sub {
      my ($source, $key, $value) = @_;
      $WordTrie->add($key);
      $WordHash->{$key} = $key;
      foreach (@{$value->{aliases} || []}) {
        exists $WordHash->{$_} and carp("Command '$key': Word '$_' already in use."), next;
        $WordTrie->add($_);
        #$CmdAliases{$_} = $key;
        $WordHash->{$_} = $key;
      }
    },
    remove_callback => sub {
      my ($source, $key, $value) = @_;
      $WordTrie->remove($key);
      delete $WordHash->{$key};
      foreach (@{$value->{aliases} || []}) {
        $WordTrie->remove($_);
        #delete $CmdAliases{$_};
        delete $WordHash->{$_};
      }
    },
    validator => sub {
      my ($source, $key) = @_;
      # if the key is in the hash and MDefList's own validation didn't catch it
      # then it's trying to overwrite an alias
      $WordHash->{$key} and croak "Cannot replace alias '$key' with command";
    },
  ),
};

Define ConnectionMethods => {
cmd_execute => sub {
  my ($con, $obj, $cmd, $args, %info) = @_;

  $con or croak "cmd_execute called without connection!";

  my $result;
  EXECUTE: {
    my $isevent = ref $cmd and $cmd->isa('MEvent');
    if (!$isevent and !$cmd_allowed->($con, $cmd)) {
      $result = [error=>{}, "You are not allowed to '$cmd'."];
      last EXECUTE;
    }
    my $cmdrec = $isevent ? {} : ($CommandInfo->get($cmd) or confess "cmd_execute called with nonexistent command!");
    my $code = $cmdrec->{code};
    
    $args =~ s/^(?:@{[join '|', @{$cmdrec->{junk_prefixes} || []}]})\b\s*//i if $args;
    
    unless ($cmdrec->{no_options}) {
      $info{"opt_$1"} = 1 while $args !~ s#^--\s*## and $args =~ s#^[-/](\w+)\s*##;
    }
    
    if (!$obj and ($code =~ /^\w+$/ or !($cmdrec->{no_object} or $cmdrec->{optional_object}))) {
      $result = [error=>{}, "You must attach to use the '$cmd' command."];
      last EXECUTE;
    }

    my $trans = MTransaction->open(connection => $con, user => $con->user);
    eval {
      if (ref $cmd and blessed($cmd, 'MEvent')) {
        $cmd->run;
      } elsif (ref $code eq 'CODE') {
        $result = $code->(($cmdrec->{no_object} ? $con : $obj), $args, %info, connection => $con);
      } elsif ($code =~ /^sub:(.*)$/) {
        $result = MDefList->path("/Subs/$1")->($con, $obj, $cmd, $args, %info);
      } elsif ($code =~ /^\w+$/) {
        $result = $obj->$code($cmd, $args, %info);
      } else {
        mudlog "ERROR/COMMANDS: Bad code for command '$cmd'";
      }
    };
    
    if (!$@) {
      $trans->commit;
      $trans->close;
    } elsif (my $cf = parse_cfail()) {
      $trans->commit;
      $trans->close;
      $result = [error=>{}, $cf];
    } else {
      my $etext = $@;
      $trans->close;
      mudlog qq{ERROR/COMMANDS: exception while running "$cmd $args" for }.($obj ? '#' . $obj->id . ' ' . ($obj->getAttr('name') || '') : 'connection '. $con->id).": \n$etext";
      $result = [error=>{}, [line=>{},"[ Errors occurred while processing your command: ]"], ['html:pre'=>{},$etext]];
    }
    
  } # :EXECUTE
  
  $con->send($result) if defined $result and (ref $result or length $result);
  return 1;
},
cmd_do => sub {
  my ($con, $obj, $line, %info) = @_;

  return unless $line;
  MScheduler::mon_push("Executing $line" . (eval {" for " . $obj->id} || "connection " . $con->id));

  if ($line =~ /^[A-Z]/) {
    $con->cmd_execute($obj, 'say', $line, %info);
    return;
  }
  
  my $first_cmd;
  eval {$first_cmd = $con->cmd_match($obj, $line)};
  if ($@) {
    if (my $cf = parse_cfail()) {
      $con->send([error=>{}, $cf]);
      return;
    } else {
      die $@;
    }
  }
  
  my $noparse = $first_cmd && $CommandInfo->get($first_cmd->[0]) && $CommandInfo->get($first_cmd->[0])->{no_parse_args};

  if ( ($line =~ s/^\(// and my $psub=1) or ($line =~ /;|[\.!?](\s|$)/ and !$noparse) ) {

    $line =~ s/\)$// if $psub;
    $line =~ s/[!?]$// if !$psub;
    
    if ((my @parts = $cmd_split->($line)) > 1) {
      foreach (@parts) {
        $con->cmd_do($obj, $_, %info);
      }
      return;
    }
  }
  
  my ($cmd, $args) = $cmd_parse->($line);
    
  if (my $rcmd = $con->cmd_match($obj, $line)) {
    $info{verb} = $rcmd->[1];
    return $con->cmd_execute($obj, $rcmd->[0], $args, %info);

  } elsif ($con and my $alias = ($con->pref('aliases') || {})->{$cmd}) { 
    my @args = split /\s+/, $args;
    $alias =~ s/\$0/$args/g;
    $alias =~ s/\$([1-9])/            $args[$1 - 1] || ''    /ge;
    $alias =~ s/\$\*([1-9])/join ' ', @args[$1 - 1 .. $#args]/ge;
    $con->send("[$alias]");
    $con->cmd_do($obj, $alias, %info);
    
  } elsif ($con and $con->pref('talk_default')) {
    cmd_execute($con, $obj, 'say', $line, %info);
    
  } else {
    $con->send([error=>{}, "No such command: '$cmd'." . ($con->pref('advice') >= 2 ? " (Type 'help' for help.)" : '')]);
    mudlog "KEYWORD: command '$cmd' '$args'";
  }
  MScheduler::mon_pop();
  return;
},
cmd_match => sub {
  my ($con, $obj, $line, %opts) = @_;

  my ($cmd, $args, $sep) = $cmd_parse->($line);

  exists $WordHash->{$cmd} and $cmd_allowed->($con, $WordHash->{$cmd}) and return [$WordHash->{$cmd}, $cmd];

  my %seen_cmds;
  my @matches = grep !$seen_cmds{$_->[0]}++ && $cmd_allowed->($con, $_->[0]), map [$WordHash->{$_}, $_], $WordTrie->lookup($cmd);

  if (@matches == 0 and !$opts{no_special} and $obj) {
    my $trans = MTransaction->open(connection => $con, user => $con->user);
    @matches = map [$_, $_->name], call_ordered_hooks('special_command', $obj, $con, $cmd, $args, $sep);
    $trans->close; # don't bother committing as this should never have side effects
  }
  if (wantarray) {
    return @matches;
  } else {
    if (@matches < 2) {
      return $matches[0];
    } else {
      my $things = join ', ', sort map "'$_->[1]'", @matches;
      $things =~ s/, ([^,]+)$/, or $1/;
      die "CFAIL:Do you mean $things?";
    }
  }
},
};

=item CON->cmd_match(OBJ, LINE, [OPTS])

In list context, returns all matching commands. In scalar context, returns one matching command
or dies if there is more than one (with a user-readable error)

The return value(s) are anonymous arrays containing [real name of command or an event, name user entered for command].

OPTS is a list of key-value options. Currently, the only defined option
is 'no_special', which prevents cmd_match from matching special commands
defined with the 'special_command' hook.

=cut

Define Commands => {
commands => {
  aliases => [qw(words verbs)],
  no_object => 1,
  code => sub {
    my ($con, $args, %info) = @_;
    
    my $alt = $info{opt_a} || $info{opt_aliases};
  
    if ($info{opt_w} or $info{opt_words} or ($info{verb}||'') eq 'words') {
      return [report=>{},
        [title=>{}, "These words will be recognized as commands"],
        ['html:ul'=>{'html:class'=>'layout'}, map ['html:li'=>{}, $_], sort grep $cmd_allowed->($con, $WordHash->{$_}), keys %$WordHash]
      ];
    } else {
      return [report=>{},
        [title=>{}, "The commands available to you are"],
        ['html:ul'=>{'html:class'=>'layout'}, map {
	  my $rec = $CommandInfo->get($_);
	  my @aliases = @{$rec->{aliases} || []};
	  ['html:li'=>{}, ($rec->{requires} ? ['html:em'=>{},$_] : $_)],
	  ($alt && @aliases ? ['html:ul'=>{'html:class'=>'linear'}, map ['html:li'=>{}, $_], @aliases] : ())
	} sort grep {
	  my $rec = $CommandInfo->get($_);
	  ($con->objects or ($rec->{no_object} or $rec->{optional_object}))
	    and $cmd_allowed->($con, $_) and !$rec->{nolist};
	} $CommandInfo->keys ],
      ];
    }
  },
  help => <<'EOHELP',
commands [-a[liases]] [-w[ords]]

Lists all the commands you can use. If the -a option is used, also lists alternate names for the commands, indented below. If the -w option is used, lists alternate names sorted with all other commands.
EOHELP
},
#---------------------------------------------------------------------------------------------------
};

### Help ##########################################################
Require 'help';
Define Help => {
  commands => {
    keywords => "entering commands input verbs",
    title => "entering commands",
    body => <<'EOT',

Commands can be abbreviated.

Commands whose names consist solely of punctuation (', ;, :) do not need a space after the name.

Since objects may be referred to by multiple words, you must include prepositions - "give key to joe" instead of "give joe key" or "give key joe". See "help info:objref" for information about how to specify exactly which object you want to manipulate.

Multiple commands may be placed on one line using ";" or ".", except after commands like "say". You can also use parentheses to group commands and override the "say" restriction:

  > say goodbye; west
  You say, "Goodbye; west."
  > (say goodbye; west)
  You say, "Goodbye."
  You leave west.
  > (say (goodbye; I will be back); west)
  You say, "Goodbye; I will be back."
  You leave west.
  
Any input starting with an uppercase letter will be assumed to be speech.
EOT
  },
};