Define '.' => {
  'Actions' => MDefList->new(),
};

###############################################################################
Require 'commands';
###############################################################################

use strict;

Define Subs => {
  TransContext => sub {
    local $Carp::CarpLevel = $Carp::CarpLevel + 1;
    return MTransaction->getContextItem(@_);
  },
  dice => sub {
    my ($num, $sides, $total) = @_;
    # number of dice, sides of dice, fixed value to add
    
    if (not defined($sides)) { # single string arg, XdY+Z
      ($num, $sides, $total) = $num =~ /^\s*(\d+)\s*d\s*(\d+)(?:\s*\+(\d+))?\s*$/;
      $num or return undef;
    }
    
    # += doesn't care whether its lvalue (l-operand?) is undef.
    while ($num-- > 0) {
      $total += int(rand($sides)) + 1;
    }
    return $total;
  },

  parse_time => sub {
    my ($str) = @_;
    #print "parse_time($str)\n";
    my $time = 0;
    my $val;
    while ($str =~ s/(-?\d+(?:\.\d+)?)([dhms]?)//) {
      ($val, my $unit) = ($1, $2);
      last unless $val;
      next if not $unit or $unit eq 's';
      $val *= 60; # minutes
      next if $unit eq 'm';
      $val *= 60; # hours
      next if $unit eq 'h';
      $val *= 24; # days
      next if $unit eq 'd';
    } continue {
      $time += $val;
    }
    return $time;
  },
  
  format_time => sub {
    my ($tyme) = @_;
    my $day = 24 * (my $hour = 60 * (my $minute = 60));
    my $days =    int($tyme / $day   ); $tyme -= $days    * $day;
    my $hours =   int($tyme / $hour  ); $tyme -= $hours   * $hour;
    my $minutes = int($tyme / $minute); $tyme -= $minutes * $minute;
    my $out = ( 
       ($days ? "${days}d" : '')
      .($hours ? "${hours}h" : '')
      .($minutes ? "${minutes}m" : '')
      .($tyme ? sprintf("%.1fs", $tyme) : '')
    );
    return $out ? $out : '0';
  },
  
  parse_cfail => sub {
    $@ and $@ =~ /^(?:# )?CFAIL:(.*?)( at .+ line \d+\.|\.?\nFile '.*'; Line \d+)\s*$/s 
       and return $1;
    return;
  },

  ThawObject => sub {
    my ($bytes) = @_;
  
    my $self = MObject->new;
  
    my $data = MFreezer::thaw($bytes);
    foreach (keys %$data) {
      $self->setAttr($_, $data->{$_});
    }
    
    $self;
  },

  max => sub {
    my $now = shift;
    foreach (@_) {
      $now = $_ if $_ > $now;
    }
    $now;
  },
  
  min => sub {
    my $now = shift;
    foreach (@_) {
      $now = $_ if $_ < $now;
    }
    $now;
  },
  
  blessed => sub {
    ref($_[0])
      ?    ( ref($_[0]) =~ /[a-z]/       )
	&& ( @_ < 2 or $_[0]->isa($_[1]) )
      : undef
  },
  

};

Define Fields => {
  article => {default => 'a'},
  name => {default => 'thing'},
  name_plural => {},
  gender => {default => 'neuter'},
  allow_genders => {default => {neuter => 1}}, # genders appropriate for this object

  type => {default => 'real'}, # real symbol virtual group

  c_hit => {default => 100},
  c_move => {default => 100},
  m_hit => {default => 0},
  m_move => {default => 0},

  has_metabolism => {default => 0},

  ldesc => {default => 'You see nothing special.'},
  idesc => {default => ''},
  
  glance_contents => {default => 0},
  enter_prep => {default => 'in'},
  sky_vis => {default => 1}, # fraction of sunlight that gets to this room - e.g. dense forest might use .1

  # Observability
  invisible => {default => 0},
  mute => {default => 0},
  blind => {default => 0},
  deaf => {default => 0},

  # Containers
  cnt_interior => {default => 0},
  cnt_type => {default => 'hard'}, # hard, soft, open
  cnt_slot_nomax => {default => 0},

  doing => {},
  do_stop => {},
  offers => {default => []},

  last_mentioned_object => {default => []},
}, Commands => {
  stop => {code => sub {
    my ($self, $args) = @_;
    if ($self->do_stop) {
      $self->do_stop->run;
      $self->nact("<self> <v:stops> ".$self->doing.".");
      $self->resetAttr('doing');
      $self->resetAttr('do_stop');
    } else {
      die "CFAIL:There's nothing for you to stop doing.";
    }
  }},
  'accept' => {code => sub {
    my ($self, $args) = @_;
    my $offers = $self->offers;
    my $offer = shift @$offers;
    $offer or die "CFAIL:There's nothing to accept!";
    $offer->{'accept'}->run;
    $self->setAttr('offers', $offers);
    return;
  }},
  deny => {code => sub {
    my ($self, $args) = @_;
    my $offers = $self->offers;
    my $offer = shift @$offers;
    $offer or die "CFAIL:There's nothing to deny!";
    $offer->{'deny'}->run;
    $self->setAttr('offers', $offers);
    return;
  }},
};

my $re_for_all_word = '^(all|every\w*)(\.|\s+|$)|(\.|\s+|^)(things|stuff|objects|items)$';

Define Methods => {

freeze => sub {
  my ($self) = @_;

  my $flist = MDefList->root->get('Fields');
  return MFreezer::freeze({map +($_, $self->getAttr($_)), grep {my $info = $flist->get($_); !$info or !$info->{nofreeze}} $self->fields});
},


can_see => sub {
  my ($self, $other) = @_;
  return 0 if MModules->loaded('body_pos') and !$self->bp_can('look');
  return 0 if $self->blind > 0;
  return 0 if $other->invisible > 0;
  1;
},

can_hear => sub {
  my ($self, $other) = @_;
  return 0 if MModules->loaded('body_pos') and !$self->bp_can('hear');
  return 0 if $self->deaf > 0;
  return 0 if $other->mute > 0;
  1;
},

vis_context => sub {
  my ($self) = @_;
  my $obj = $self->getAttr('container');
  $obj = $obj->getAttr('container') while $obj->getAttr('container') and $obj->getAttr('glance_contents');
  return $obj;
},

object_scan => sub {
  my ($self, $hook, %param) = @_;
  
  if (!MModules->loaded('containment')) {
    mudlog "DEBUG: object_scan reject";
    return $self;
  }

  $hook->($self) and return unless $param{no_self};
  unless ($param{no_self_contents} || (($param{_nsc_exclude}||0) == $self)) {
    foreach my $obj (@{$self->contents}) {
      next if $param{_only_symbols} and $obj->getAttr('type') ne 'symbol';
      $hook->($obj) and return;
    }
    foreach my $obj (@{$self->contents}) {
      next if $param{_only_symbols} and $obj->getAttr('type') ne 'symbol';
      if ($obj->glance_contents and !$param{no_descend}) {
        $obj->object_scan($hook, no_outside => 1, no_self => 1, _nsc_exclude => $param{_nsc_exclude});
      }
    }
  }  
  if (!$param{no_outside} and my $obj = $self->getAttr('container')) {
	while ($obj->glance_contents) {
	  $obj = $obj->container || last;
	}
	# after we find the outermost visible container(s), we descend thru the tree.
	$obj->object_scan($hook, no_outside => 1, _nsc_exclude => $self, no_descend => $param{no_descend});
  }
  if ($param{symbols}) {
    (ObjectByName('symbols') || return)->object_scan($hook, _only_symbols => 1, no_outside => 1);
  }
},

# *** NOTE: It is the responsibility of the calling routine to determine
# if the object(s) returned by object_find are physically accessible.
object_find => sub {
  my ($self, $name, %param) = @_;
  
  my $caller = $param{'caller'} || $self;

  my $orig_name = $name || '';
  $name = defined $name ? lc $name : '';
  my $had_an = $name =~ s/\s*\ban?\b\s*//g;
  $name =~ s/\.$//;

  die "CFAIL:" . ucfirst( ($param{verb} ? $param{verb} . ' ' : '') . ($had_an ? "a what?" : "what object?") ) if !$name;

  if ($name =~ /\band\b|,/) {
    my @res = map {eval {$self->object_find($_, %param)}} grep $_, split /(?:\s*(?:\band\b|,))+\s*/, $name;
    if ($@) {
      if (my $cf = parse_cfail()) {
        TransContext('connection')->send([error=>{}, $cf]);
      } else {
        die $@;
      }
    }
    if (@res) {
      $self->last_mentioned_object([@res]);
      return wantarray ? @res : $res[0];
    } else {
      die "CFAIL:" . ucfirst( ($param{verb} ? $param{verb} . ' ' : '') . ($had_an ? "a what?" : "what object?") );
    }
  }
  
  my $isf;
  if ($name =~ /^(.*'s|its|his|her|their)\s(.*)$/ or $isf = $name =~ /^(.*)\s+(?:from|in|on|of)\s+(.*)$/) {
    my ($from, $thing) = $isf ? ($2, $1) : ($1, $2);
    #mudlog "finding >$thing< from >$from<";
    $from =~ s/'?s$//;
    my %nok = map {$_, 1} qw(no_self_contents no_outside);
    my %restricted_param = map {$_, $param{$_}} grep !$nok{$_}, keys %param;
    return $self->object_find(
      $from,
      %restricted_param
    )->object_find(
      $thing,
      'caller' => $caller,
      %restricted_param,
      ($thing =~ $re_for_all_word ? (no_outside => 1, no_self => 1) : ())
    );
  }

  return $self if $name =~ /^(?:me|(?:my)?self)$/;

  if (TransContext('user') and TransContext('user')->privileged('watcher')) {SPECIAL_REF: {
    my $obj;
    if ($name =~ /^#(\d+)$/) {
      my $id = $1;
      $obj = ObjectByID($id) or die "CFAIL:There is no object with ID $id.";
    } elsif ($orig_name =~ /^\s*\$(\S+)/) {
      my $oname = $1;
      $obj = ObjectByName($oname) or die "CFAIL:There is no object named '$oname'.";
    } else { 
      last SPECIAL_REF;
    }
    $self->last_mentioned_object([$obj]);
    return $obj;
  }}

  if ($name =~ /^(it|them|him|her)$/i) {
    my $word = $1;
    my @found = grep {$_->ref_exists} my @all_it = @{ $self->last_mentioned_object };
    @all_it or die "CFAIL:What's ".lc($word)."?";
    
    # FIXME: we're handing out information here we shouldn't be.
    # 'it' should be restricted to only objects you can currently see
    # and if you can't, it gives the same message as the object not existing
    @found or die "CFAIL:".ucfirst(lc $word)." isn't around anymore.";
    
    return wantarray ? @found : $found[0];
  }
  
  $param{no_outside} = 1 if $name =~ s/^my\s+//;
  $param{no_self_contents} = 1 if $name =~ s/^the\s+// and !$param{no_outside};
  my $instance = ($name =~ s/$re_for_all_word// || $name =~ s/e?s$//) ? -1
               : ($name =~ s/^(\d+)(?:\.|(?:st|nd|rd)\s+)// ? $1 : 0)
              || ($name =~ s/\.(\d+)$// ? $1 : 0)
              || ($name =~ s/\s+(\d+)$// ? $1 : 0)
              || 1;
  my $orig_instance = $instance;
  $param{no_outside} = 1 if $name =~ s/^my\s+//;
  $param{no_self_contents} = 1 if $name =~ s/^the\s+// and !$param{no_outside};
  # my/the checks twice to handle "my all" as well as "all my food", etc.
 
  if ($instance == -1) {     # if this is an 'all'...
    $param{no_self} = 1;     # never include self
    $param{no_descend} = 1 if !$name; # if this is an unqualified 'all', don't get contents of things
  }
 
  #$caller->send("DEBUG: name after processing: '$name', instance: $instance");
  #$caller->send("Params: " . join ', ', %param);
  
  my @found;
  $self->object_scan(sub {
    my ($obj) = @_;
    return 1 if $instance == 0;
    
    return unless $caller->can_see($obj);
    
    return if ($caller->container || 0) == $obj and $instance == -1;
    #$caller->send("DEBUG: in scan callback, scanning #".$obj->id.", instance now is $instance");
    
    push @found, $obj if  ( !$name or $obj->name =~ /\b\Q$name\E/i 
                                   or ($instance == -1 and ($obj->name_plural||'') =~ /\b\Q$name\E/i) )
                      and ( $instance == -1 ? $obj != $caller : --$instance == 0 );
    # if $instance == -1, then we're scanning "all", therefore never include self.
                         
    return 1 if $instance == 0;
  }, %param);
 
  if (@found) {
    $self->last_mentioned_object([(wantarray ? @found : $found[0])]);
    return wantarray ? @found : $found[0];
  } else {
    use Lingua::EN::Inflect ();
    # FIXME: this doesn't fit viewpoints correctly
    die "CFAIL:".($caller->id==$self->id?"You don't ":ucfirst$self->nphr." doesn't ")
      . ($param{no_outside} ? 'have ' : 'see ') 
      . ($instance == -1 ? (length($name) ? qq{any $name} : 'anything')
                         : (length($name) ? Lingua::EN::Inflect::A($name) : 'a thing')
                         ) 
      . ($orig_instance > 1 ? " #$orig_instance" : '')
      . ($param{no_outside} ? '' : ' ' . ($caller->container->id == $self->container->id ? 'here' : $self->enter_prep . ' ' . $self->nphr))
      . '.';
  }
},

};

Define Hooks => {

'special_command' => sub {
  my ($self, $con, $cmd, $args, $sep) = @_;
  
  my @obj = eval {$self->object_find($cmd.$sep.$args)};
  
  if ($@) {
    $@ !~ /CFAIL/ and die $@;
    return;
  }

  return MEvent::Message->new(
    target => $self,
    method => 'do_nouns',
    arguments => \@obj,
  );
},

prompt_info => sub {
  my ($self) = @_;
  return () unless $self;
  return (
    $self->invisible ? 'invis' : (),
#    int($self->c_hit / $self->m_hit * 100) . "H",
#    int($self->c_move / $self->m_move * 100) . "V",
  ) if $self;
},

};

Define Help => {
objref => {title => 'referring to objects', keywords => 'objects reference select name my the all', body => xml2sx(<<'EOT')},
<group>
<html:p>Let's use as an example that there is a wooden box on the floor of the room you are in, and you want to pick it up. The simplest way to do so would be to type:</html:p>

<html:blockquote><html:pre>&gt; get box</html:pre></html:blockquote>

<html:p>But what if there is also an iron box? Then, you need to specify what you want:</html:p>

<html:blockquote><html:pre>&gt; get wooden box</html:pre></html:blockquote>

<html:p>Also, provided there are no other 'wooden' objects, you can type "<html:code>get wooden</html:code>" to get the box.</html:p>

<html:p>If there are two identical objects visible, and you want to get a specific one, you can refer to it by number: "<html:code>get wooden box 2</html:code>". If you wish to manipulate multiple objects, you can use "<html:code>get all</html:code>" or "<html:code>get all wooden box</html:code>".</html:p>

<html:p>If you want to refer specifically to something you are carrying, you can use 'my', for example "<html:code>look at my wooden box</html:code>" to make sure you look at the box you're carrying, as opposed to the one in the room with you. "The" functions equivalently for things you are NOT carrying.</html:p>

<html:p>You can combine "my", "the", and "all", though "my the" will never match anything. Also, the words "things", "stuff", "objects", or "items" can be used AFTER the object name to do the same thing as "all" before, e.g. "<html:code>drop my iron stuff</html:code>".</html:p>
</group>
EOT
};