Require 'privcore', 'commands';
Require 'powers'; # is this needed?


my $edit_target_obj = sub {
  my $con = {@_[2..$#_]}->{connection}; # sneaky, we call this sub with no-parens & syntax, thus inheriting @_
  my $t = $con->pref('edit_target');
  defined $t or do {
    die "CFAIL:No edit target selected.";
  };
  my $obj = ObjectByID($t);
  defined $obj or do {
    $con->pref('edit_target', undef);
    die "CFAIL:Edit target no longer exists.";
  };
  $obj;
};

Define Fields => {
  'owner' => {noinherit => 1},
}, Commands => {
#---------------------------------------------------------------------------------------------------
'reuse' => {
  requires => [qw(controller)],
  no_object => 1,
  code => sub {
    my ($con, $args) = @_;
    $args or die "CFAIL:Usage: reuse <Perl module>";
    my $re = $args;
    $re =~ s/::/\\W+/;
    foreach (grep {$_ =~ /$re/} keys %INC) {
      delete $INC{$_};
    }
    my $warnbuf = '';
    local $SIG{__WARN__} = sub {$warnbuf .= $_[0] unless $_[0] =~ /[Ss]ubroutine \w+ redefined|Ambiguous use of (\{|\w+ =>)/};
    eval "use $args";
    my $err = $@;
    mudlog "(PC) @{[$con->user->name]} re-used $args" . ($@ ? ":\n$@" : '');
    $warnbuf . ($err || '');
  },
  help => <<'EOHELP',
reuse <Perl module>

Reloads a given Perl module. &:sb;This command should NOT be used unless you are familiar with the operation of Perl and the code of mpMUD.&:n 

Important to note is that this will not cause new imported symbols to be visible to existing packages; it only reloads the code. 

This command should only be used in emergency situations.
EOHELP
},
#---------------------------------------------------------------------------------------------------
'sync' => {
  requires => [qw(builder)],
  no_object => 1,
  code => sub {MObjectDB->sync},
  help => <<'EOHELP',
Forces the object cache, scheduled event list, and user data to be written to disk.
EOHELP
},
#---------------------------------------------------------------------------------------------------
'attributes' => {
  requires => [qw(watcher)],
  no_object => 1,
  code => sub {
    my ($con, $args) = @_;
    
    if ($args) {
      return sx_dump_thingy(MDefList->root->get('Fields')->get($args));
    } else {
      return [report=>{}, ['html:ul'=>{'html:class'=>'report'}, map ['html:li'=>{},$_], sort MDefList->root->get('Fields')->keys]];
    }
  },
  help => <<'EOHELP',
attributes [&:meta<name>&:n]

Displays all globally defined object attributes or, with an argument, the flags / default value of a specific field. This command needs improvement.
EOHELP
},
#---------------------------------------------------------------------------------------------------
'names' => {
  requires => [qw(watcher)],
  no_object => 1,
  code => sub {
    my ($con, $args) = @_;
    
    [report=>{},
      [title=>{}, 'Named objects'],
      ['html:ul'=>{}, map ['html:li'=>{'html:class'=>'report'},$_], sort MObjectDB->names]
    ];
  },
},
#---------------------------------------------------------------------------------------------------
'methods' => {
  requires => [qw(watcher)],
  no_object => 1,
  code => sub {
    my ($con, $args) = @_;
    
    #FIXME: direct hash access
    [report=>{},
      [title=>{}, "MObject's instance methods"],
      ['html:ul'=>{'html:class'=>'report'}, map ['html:li'=>{},$_], sort keys %MObject::VisInstanceMethods]
    ];
  },
},
#---------------------------------------------------------------------------------------------------

### Target Editing Commands #####################################################################################################

#---------------------------------------------------------------------------------------------------
'@select' => {
  aliases => [qw(@edit)],
  requires => [qw(builder)],
  code => sub {
    my ($self, $args, %info) = @_;
    my $con = $info{connection};

    my $noview = $info{opt_s} || $info{opt_short};
    my $target;
    if ($args) {
      $target = $self->object_find($args, entire_world => 1);

      edit_priv_assert($target); # check privileges
	
      $con->pref(edit_target => $target->id);
      $target->send("You feel as if something is watching you.");
      $con->cmd_execute($self, '@view', '') unless $noview;
    } else {
      $target = &$edit_target_obj;
    }
    return [report=>{}, "Edit target" . ($args ? ' selected' : '') . ": ", [obj=>{part=>'idn'},$target], "."];
  },
  help => <<'EOHELP',
@select [-short] &:g;thing&:n;

The &:y;@select&:n; command is used to select objects for editing. The argument can be any normal means of referring to an object.

The -s[hort] flag causes &:y;@select&:n; to not call &:y;@view&:n; after selecting the target.

If you use the "@edit" alias for this command, you will be placed in attribute-editing mode.
EOHELP
},
#---------------------------------------------------------------------------------------------------
'@view' => {
  aliases => [qw(stat)],
  requires => [qw(watcher)],
  optional_object => 1,
  code => sub {
    my ($self, $args, %info) = @_;
    my ($tname, $target);

    if ($args && $self) {
      $target = $self->object_find($args, entire_world => 1);
    } else {
      $target = &$edit_target_obj;
    }
    
    return [report=>{},
      [title=>{}, "(#".$target->id.") ", ($target->locked ? "(locked) " : ()), [obj=>{part=>'poss'}, $target], " attributes"],
      ['html:ul'=>{'html:class'=>'layout'},
        map ['html:li'=>{},
          [ipre=>{}, sprintf("%12s", $_)],
          ": ",
          sx_dump_thingy($target->getAttr($_))
        ], sort $target->attributes
      ],
    ];
  },
  help => <<'EOHELP',
@view <thing>
@view

@view allows you to examine the attributes of an object. Without an argument, it displays the currently selected object. With an argument (which is interpreted the same way as by @select), it displays that object, even if you do not have permission to select that object for editing.
EOHELP
},
#---------------------------------------------------------------------------------------------------
'@attribute' => {
  aliases => [qw(@get @set)],
  requires => [qw(builder)],
  no_object => 1,
  no_parse_args => 1,
  code => sub {
    my ($con, $args, %info) = @_;

    my @args = split /\s*=\s*|\s+/, $args, 2;
    if (@args == 0) {
      return if $info{'@attr_dont_enter_mode'};
      $con->input_mode_push(Attr => sub {
        my ($con, $input) = @_;
        if ($input =~ /^@/) {
          $con->cmd_do($con->object, $input);
        } else {
          $con->cmd_execute($con->object, '@attribute', $input, '@attr_dont_enter_mode' => 1);
        }
        return 'continue';
      });
      return;
    }
    
    my $obj = &$edit_target_obj;
    my $f_xml = $info{opt_x} || $info{opt_xml};
    my $key = $args[0];
    my $nphr = $obj->nphr; # get name NOW so if name changes, message uses old name
    my $is_setting = @args > 1;
    
    if ($is_setting) {
      my $value = $args[1];

      $value =~ s/\n/\cJ/g; # so it's at least always the same
  
      # this ought to be replaced with a general value-parsing routine
      if ($f_xml) {
        $value = xml2sx($value);
      } elsif ($value =~ s/^(['"])(.*)\1/$2/) {
        # do nothing (quoted string)
      } elsif ($value =~ s/^\[(.*)\]/$1/) {
        $value = [split /\s*,\s*/, $value];
      } elsif ($value =~ s/^\{(.*)\}/$1/) {
        $value = {split /\s*(?:,|=>|:)\s*/, $value};
      } elsif ($value =~ s/^[#\$]//) {
        $value = ($info{object} or die "CFAIL:No object to parse object reference by.")->object_find($value, entire_world => 1);
      }

      if (my $fattrs = MDefList->root->get('Fields')->get($key)) {
      
        $fattrs->{noset} and die "CFAIL:That attribute shouldn't be directly set.";
  
        if (!ref $value) {
          if    (ref($fattrs->{default}) eq 'ARRAY') { $value = [split /\s+/, $value]; }
          elsif (ref($fattrs->{default}) eq 'HASH' ) { $value = {map {$_, 1} split /,/, $value}; }
        }
      } else {
        $key =~ /^_/ and die "CFAIL:You can't set attributes starting with underscores."; 
        $con->send([error=>{}, "Warning: $key is a non-standard attribute."]);
      }
  
      eval { $obj->setAttr($key, $value); };
      $@ and return [error=>{}, ['html:pre'=>{}, $@]];
    }

    my $proc = $f_xml ? sub {['html:pre'=>{}, sx2xml($_[0])]} : sub {sx_dump_thingy(@_)};
    
    return [report=>{}, [obj=>{}, "${nphr}'s"], " $key".($is_setting?" set to: " : ": "), $proc->($obj->getAttr($key), heredoc => ($info{opt_h} || $info{opt_heredoc}), get => $is_setting)];
  },
  help => xml2sx(<<'  EOHELP'),
  <group>
    <html:p>The @attribute command is used for nearly all attribute editing. Usage:</html:p>
    <html:ul>
      <html:li>@attr: enters attribute-editing mode; see below.</html:li>
      <html:li>@attr <meta>&lt;attr&gt;</meta>: displays the value of that attribute.</html:li>
      <html:li>@attr <meta>&lt;attr&gt;</meta> <meta>&lt;value&gt;</meta>: sets the attribute to the <meta>&lt;value&gt;</meta>. A subset of Perl syntax is available for entering structures.</html:li>
    </html:ul>
  </group>
  EOHELP
},
#---------------------------------------------------------------------------------------------------
'@replace' => {
  requires => [qw(builder)],
  no_object => 1,
  code => sub {
    my ($con, $args) = @_;
    my $obj = &$edit_target_obj;
    
    $args =~ s/^(\S+)\s+(.)// or die "CFAIL:Usage: \@replace <field> /<old>/<new>/<opts>";
    my ($field, $delim) = ($1, $2);
    my ($old, $new, $mod) = split /\Q$delim/, $args, 3;
    $mod ||= '';
    my $text = $obj->getAttr($field);
    eval {$text =~ s/(?$mod)$old/$new/};
    $con->send($@||'');
    $con->cmd_execute(undef, '@set', "$field=$text");
  },
  help => <<'EOHELP',
@replace allows you to perform a text replacement on an attribute; this is useful for editing object descriptions.

@replace <field> /old/new/

where 'old' is a regular expression.

You can use any character you like in place of the /.
EOHELP
},
#---------------------------------------------------------------------------------------------------
'@unset' => {
  aliases => [qw(@clear)],
  requires => [qw(builder)],
  no_object => 1,
  code => sub {
    my ($con, $key) = @_;
    my $obj = &$edit_target_obj;
    
    my $fattrs = MDefList->root->get('Fields')->get($key);
    $fattrs->{noset} and die "CFAIL:That attribute can't be directly set.";

    eval { $obj->resetAttr($key) };
    $@ and return [error=>{}, ['html:pre'=>{}, $@]];
    [report=>{}, [line=>{}, [obj=>{part=>'poss'}, $obj], " $key deleted."]];
  },
  help => <<'EOHELP',
@clear key

Deletes a field in the currently selected object, causing the value to be inherited from the object's prototype.
EOHELP
},
#---------------------------------------------------------------------------------------------------
((eval "use MIME::Base64 ()", $@) ? () : ('@freeze' => {
  requires => [qw(builder)],
  no_object => 1,
  code => sub {
    my ($con, $args) = @_;
    my $obj = &$edit_target_obj;
    
    return <<"EOTHING";
Content-Type: application/octet-stream; name="@{[$obj->name]}#@{[$obj->id]}.obj"
Content-Transfer-Encoding: base64

@{[MIME::Base64::encode($obj->freeze)]}
EOTHING
  },
})),
#---------------------------------------------------------------------------------------------------
'@chown' => {
  requires => [qw(builder)],
  code => sub {
    my ($self, $args) = @_;
    my $obj = &$edit_target_obj;

    MUser->exists($args) or die "CFAIL:\@chown: no such user: $args";

    $obj->setAttr('owner', $args);
    [action=>{}, [obj=>{part=>'poss'}, $obj], " owner set to $args."];
  },
  help => <<'EOHELP',
@chown &:meta;<owner>&:n;

Sets the owner of the currently selected object. After the owner has been changed, you can still edit the object, but you cannot @select it again.
EOHELP
},
#---------------------------------------------------------------------------------------------------
'@name' => {
  requires => [qw(builder)],
  no_object => 1,
  code => sub {
    my ($con, $args) = @_;
    my $obj = &$edit_target_obj;

    if (length $args) {
      $args =~ s/^\$//;
      $obj->addName($args);
      [action=>{}, "Name '$args' added to ", $obj, "."];
    } else {
      my @names = $obj->getNames;
      
      [report=>{}, (@names
        ? ([title=>{}, "Names for ",$obj], [list=>{}, map [li=>{}, $_], @names])
        : ("There are no names for ",$obj,".")
      )];
    }
  },
  help => <<'EOHELP',
@name
@name &:meta;<name>&:n;

Lists the names for the currently selected object, or adds a new name for it.
EOHELP
},
#---------------------------------------------------------------------------------------------------
'@unname' => {
  requires => [qw(builder)],
  no_object => 1,
  code => sub {
    my ($con, $args) = @_;
    my $obj = &$edit_target_obj;

    $args =~ s/^\$//;
    $obj->delName($args);
      [action=>{}, "Name '$args' removed from ", $obj, "."];
  },
  help => <<'EOHELP',
@unname &:meta;<name>&:n;

Sets a name for the currently selected object.
EOHELP
},
#---------------------------------------------------------------------------------------------------
'@lock' => {
  requires => [qw(builder)],
  no_object => 1,
  code => sub {
    my ($con, $args) = @_;
    my $obj = &$edit_target_obj;

    $obj->lock;
    [report=>{}, $obj, ' locked.'];
  },
  help => 'Locks the currently selected object.',
},
#---------------------------------------------------------------------------------------------------
'@unlock' => {
  requires => [qw(builder)],
  no_object => 1,
  code => sub {
    my ($con, $args) = @_;
    my $obj = &$edit_target_obj;

    $obj->unlock;
    [report=>{}, $obj, ' unlocked.'];
  },
  help => 'Unlocks the currently selected object.',
},
#---------------------------------------------------------------------------------------------------
};

Define Subs => {
edit_target_obj => $edit_target_obj,
edit_priv_assert => sub {
  my ($tob) = @_;
  
  my $user = TransContext('user');
  $user->privileged('controller')       and return 1; # Controllers can edit anything
  not defined (my $owner = $tob->owner) and return 1; # No owner, you can edit it
  lc $user->name eq lc $owner           and return 1; # You're the owner, you can edit it
  
  # You're not a controller and not the owner, so you can't edit it
  die 'CFAIL:You do not have permission to edit #'.$tob->id.'.';
},
};
