Require 'commands';

my %fields = (
  'width'   => {type => 'int', key => 'scr_width',  'min' => 12, 'max' => 400, mpre => ['Your screen ', [key=>{}, 'width'], ' is']},
  'height'  => {type => 'int', key => 'scr_height', 'min' => 4, 'max' => 2000, mpre => ['Your screen ', [key=>{}, 'height'], ' is']},
  'color'   => {type => 'op!', key => 'ansi_off',                              mpre => [                [key=>{}, 'Color'], ' is']},
  'brief'   => {type => 'opt', key => 'brief',                                 mpre => [                [key=>{}, 'Brief'], '-look mode is']},
  'prompt'  => {type => 'opt', key => 'prompts',                               mpre => ['Command ',     [key=>{}, 'prompt'], 's are']},
  'speech'  => {type => 'opt', key => 'talk_default',                          mpre => ['Unrecognized commands as ', [key=>{}, 'speech'], ' is']},
  'filter'  => {type => 'opt', key => 'speech_filter',                         mpre => ['Speech ',      [key=>{}, 'filter'], 'ing is']},
  'chat'    => {type => 'opt', key => 'show_channels',                         mpre => [                [key=>{}, 'Chat'], ' messages are']},
  'advice'  => {type => 'lev', key => 'advice', 'min' => 0, 'max' => 3,        mpre => [                [key=>{}, 'Advice'], ' level is']},
  'viewpoint'  => {type => 'lev', key => 'viewpoint', 'min' => 1, 'max' => 3,  mpre => [                [key=>{}, 'Viewpoint'], ' is']},
);

my %special_fields = (
  'log' => {type => 'opt', key => 'mudlog', mpre => [[key=>{}, 'Log'], ' messages are']},
);

Define Hooks => {
'special_command' => sub {
  my ($self, $con, $cmd, $args) = @_;
  return unless $fields{$cmd};
  return MEvent::Message->new(
    target => $con,
    method => 'cmd_execute',
    arguments => [$self, 'cpref', "$cmd $args", connection => $con],
    no_store => 1,
  );
},
};


Define PrefDefaults => {
  show_channels => 1,
  advice => 3,
};

Define Commands => {
#---------------------------------------------------------------------------------------------------
  cpref => {aliases => [qw(preference config toggles options)], no_object => 1, code => sub {
    my ($con, $args, %info) = @_;
    
    my ($field, $value) = split /\s+/, $args;
    $value = undef if defined $value and $value eq '';

    my $rep = [report=>{}, ($field ? () : [title=>{}, "Connection preferences"]), my $list = ['html:ul'=>{$field ? ('html:class'=>'layout') : ('html:class'=>"structure")}]];
    
    my %my_fields = (%fields, ($con->user and $con->user->privileged('watcher')) ? %special_fields : ());
    
    my @info = $field ? ($my_fields{$field} || die "CFAIL:Unknown preference.") : map $my_fields{$_}, sort keys %my_fields;
    
    foreach my $info (@info) {
      if (defined $value) {
        if ($info->{type} eq 'int') {
          $value = int $value;
          if ($value < $info->{'min'}) {
            $value = $info->{'min'};
            $con->send([error=>{}, "$value is below the minimum value, $info->{'min'}."]);
          }
          if ($value > $info->{'max'}) {
            $value = $info->{'max'};
            $con->send([error=>{}, "$value is above the maximum value, $info->{'max'}."]);
          }
          #$value = $info->{'max'} if $value > $info->{'max'};
        } elsif ($info->{type} eq 'lev') {
          $value = 1 if $value =~ /on|yes|true/;
          $value = 0 if $value =~ /off|no|false/;
          $value = int $value;
          $value = $info->{'min'} if $value < $info->{'min'};
          $value = $info->{'max'} if $value > $info->{'max'};
        } elsif ($info->{type} eq 'opt' or $info->{type} eq 'op!') {
          my $t;
          $t = 1 if $value =~ /on|yes|1|true/;
          $t = 0 if $value =~ /off|no|0|false/;
          $value = ($t xor $info->{type} eq 'op!');
        } else {
          mudlog "ERROR: Unknown pref type in cpref set";
          die "CFAIL:Sorry, an internal error occured.";
        }
        $con->pref($info->{key}, $value);
      }
      my $val;
      if    ($info->{type} eq 'int') { $val = $con->pref($info->{key}) || 0;                }
      elsif ($info->{type} eq 'lev') { $val = ($con->pref($info->{key}) || 0) . " of $info->{min}-$info->{max}"; }
      elsif ($info->{type} eq 'opt') { $val = $con->pref($info->{key}) ? 'on' : 'off'; }
      elsif ($info->{type} eq 'op!') { $val = $con->pref($info->{key}) ? 'off' : 'on'; }
      else {
        mudlog "ERROR: Unknown pref type in cpref display";
        die "CFAIL:Sorry, an internal error occured.";
      }  
      push @$list, ['html:li'=>{}, @{$info->{mpre}}, (defined $value ? ' now' : '')." $val."];
    }
    return $rep;
  }, help => <<'EOHELP'},
cpref [ width | height | color | brief | prompt | combine | speech ] [&:meta<value>&:n]

For viewing and setting connection options. Use "&:y;cpref <field>&:n" to see the current value, "&:y;cpref <field> <value>&:n" to change it, or "&:y;cpref&:n;" by itself to see all preferences.

Values are:
  &:y;width&:n; (number): Terminal width
  &:y;height&:n; (number): Terminal height
  &:y;color&:n; (toggle): Color on
  &:y;brief&:n; (toggle): Brief mode - omit some detailed descriptions.
  &:y;prompt&:n; (toggle): Display a prompt.
  &:y;speech&:n; (toggle): Any input lines not understood as commands will be spoken.
EOHELP
#---------------------------------------------------------------------------------------------------
terminal => {no_object => 1, code => sub {
  my ($con, $args, %info) = @_;
  $con->terminal("MTerminal::\U$args"); # FIXME: need more security 
  return [report=>{}, "Terminal type set."];
}},
#---------------------------------------------------------------------------------------------------
chat => {no_parse_args => 1, no_object => 1, aliases => [','], code => sub {
  my ($con, $args, %info) = @_;
  
  my $user = $con->user;
  $con->pref('show_channels') or die "CFAIL:You can't chat while you have chat disabled - try 'cpref chat on'";
  my $sep = $args =~ s/^:// ? $args =~ /\W/ ? '' : ' ' : ': ';
  for (grep $_->pref('show_channels') && $_->state =~ /command|talk/, MConnection->all) {
    $_->send([action=>{}, '[Chat] ', ($user ? $user->name : 'guest '.$con->id), $sep, [user=>{}, $args]]);
  }
  return;
}},
#---------------------------------------------------------------------------------------------------
whoami => {no_object => 1, code => sub {
  my ($self, $args, %info) = @_;
  
  my $user = $self->user;
  my $char = $user ? $user->get('object') : undef;
  my $attached = $self->object;
  [report=>{},
    [line=>{}, "Logged in as " . ($user ? $user->name : 'a guest'),
      ($char && $char->ref_exists ? (" with character '", [obj=>{}, $char->nphr], "'") : ()), 
      ($attached && (!$char or $char->id != $attached->id) ? ("; attached to '", [obj=>{}, $attached->nphr], "'") : ()), 
      '.'
    ]
  ];
}},
#---------------------------------------------------------------------------------------------------
title => {
  no_object => 1,
  code => sub {
    my ($con, $args, %info) = @_;
    $args =~ s/^(?=[\w(])/ /;
    $con->user->set('title', $args);
    "Okay, you're now " . $con->user->name . "$args.\n";
  },
  help => <<'EOHELP',
title &:meta;<text>&:n;

Sets your "title" as displayed in the 'who' list.
EOHELP
},
#---------------------------------------------------------------------------------------------------
report => {optional_object => 1, code => sub {
  my ($self, $args, %info) = @_;
  
  my ($type, $str) = split /\s+/, $args, 2;

  $type and $type =~ /^(bug|idea|typo|note|gripe)$/i
     or return [error=>{}, "Usage: report bug|idea|typo|note|gripe ", [meta=>{}, "<message>"]];

  mudlog uc($type) . " REPORT: "
    . $info{connection}->user->name . ($self ? " (" . $self->nphr . " at " . $self->container->name . "#" . $self->container->id : "")
    . ": $str";
  "Thank you for the \L$type\E report.";

}, help => <<'EOHELP'},
report bug|idea|typo|note|gripe &:g;<message>&:n;

This command is used to report problems and ideas to the administration. Your &:g;<message>&:n; is logged, together with the room which you are in.
EOHELP
#---------------------------------------------------------------------------------------------------
think => {code => sub {$_[0]->send([action=>{}, '*', [user=>{}, $_[1]], '*'])}},
#---------------------------------------------------------------------------------------------------
alias => {no_object => 1, code => sub {
  my ($con, $args, %info) = @_;
  
  my ($name, $cmds) = split /\s+/, $args, 2;
  
  my $atab = $con->pref('aliases');
  $atab or $con->pref('aliases', $atab = {});
  
  if (defined $name) {
    if (defined $cmds) {
      $atab->{$name} = $cmds;
      return [report=>{}, "Set alias."];
    } else {
      delete $atab->{$name};
      return [report=>{}, "Deleted alias."];
    }
  } else {
    return [report=>{},
      [title=>{}, "Currently defined aliases"], 
      [list=>{},
        map [li=>{}, [pre=>{}, sprintf("%-16s -> %s", $_, $atab->{$_})]], keys %$atab
      ],
    ];
  }
}, help => <<'EOHELP'},
alias
alias &:meta;<word> <commands>&:n;
alias &:meta;<word>&:n;

Aliases allow you to substitute short commands for long ones, e.g. "kw" becomes "kill wolf".

You can make use of arguments to the alias by the replacements '$0' through '$9'. $0 is the entire argument string, and $1-$9 are individual words. You can also use $*&:c;<n>&:n; to get all the words from &:c;<n>&:n; to the end of the line, where &:c;<n>&:n; is a number from 1 to 9.

If no &:meta;<commands>&:n; are specified, the alias is deleted.

Example:
  > alias bsay emote says boredly, "$0"
  Set alias.
  > bsay So what?
  Marn says boredly, "So what?"
EOHELP
#---------------------------------------------------------------------------------------------------
review => { code => sub {
  my ($self, $args, %info) = @_;

  my $obj = $self->object_find($args);
  (my $con) = grep $_->can('captured'), MConnection->obj_connections($obj)
    or return [error=>{}, $obj, " has no log to review."];
  [report=>{}, [title=>{}, "Log of ",$obj], $con->captured];
}},
#---------------------------------------------------------------------------------------------------
list => { code => sub {
  my ($self, $args) = @_;

  return [report=>{},
    [title=>{}, 'You see'],
    ['html:ul'=>{'html:class'=>'report'}, 
       map ['html:li'=>{}, $_, ($_->container ? (' in ', $_->container) : ())], $self->object_find("all $args")
    ],
  ];
}, help => <<'EOHELP'},
list &:meta;<object>&:n;

Lists every &:meta;<object>&:n; you can see.
EOHELP
#---------------------------------------------------------------------------------------------------
'after' => {
  no_parse_args => 1,
  code => sub {
    my ($self, $args, %info) = @_;
    my ($time, $cmd) = split /\s+/, $args, 2;
    my $id = $self->id;
    MEvent::Message->new(
      name => "ST:After $time: $cmd",
      owner => $self,
      'time' => 0+parse_time($time),
      is_real_time => 1, #FIXME
      target => $self,
      method => 'do',
      arguments => [$cmd],
    )->schedule;
    return;
  },
  help => 'after <time> <command>
Executes a command later.',
},
#---------------------------------------------------------------------------------------------------
date => {aliases => ['time'], no_object => 1, code => sub {scalar localtime}},
#---------------------------------------------------------------------------------------------------
};

